<?php

namespace WPForms\Admin\Blocks;

/**
 * Class for rendering links in the admin area.
 *
 * @since 1.9.7
 */
class Links {

	/**
	 * Render links.
	 *
	 * @since 1.9.7
	 *
	 * @param array $utm_params UTM parameters to append to links.
	 */
	public static function render( array $utm_params ): void {

		$links = self::get_links( $utm_params );

		echo '<div class="wpforms-links">';

		foreach ( $links as $slug => $link ) {
			self::render_link( $slug, $link );
		}

		echo '</div>';
	}

	/**
	 * Render a single link.
	 *
	 * @since 1.9.7
	 *
	 * @param string $slug The slug for the link.
	 * @param array  $link The link data.
	 */
	private static function render_link( string $slug, array $link ): void {

		$url    = $link['url'] ?? '#';
		$text   = $link['text'] ?? '';
		$class  = $link['class'] ?? '';
		$target = $link['target'] ?? '_self';
		$icon   = $link['icon'] ?? '';

		printf(
			'<a href="%1$s" target="%2$s" rel="noopener noreferrer" class="wpforms-link wpforms-link-%3$s %4$s">%6$s%5$s</a>',
			esc_url( $url ),
			esc_attr( $target ),
			esc_attr( $slug ),
			esc_attr( $class ),
			esc_html( $text ),
			wp_kses(
				$icon,
				[
					'svg'  => [
						'xmlns'   => true,
						'width'   => true,
						'height'  => true,
						'viewbox' => true,
						'fill'    => true,
					],
					'path' => [
						'd'    => true,
						'fill' => true,
					],
				]
			)
		);
	}

	/**
	 * Get links.
	 *
	 * @since 1.9.7
	 *
	 * @param array $utm_params UTM parameters to append to links.
	 *
	 * @return array
	 */
	private static function get_links( array $utm_params ): array {

		return [
			'docs'      => [
				'url'    => self::get_utm_link(
					'https://wpforms.com/docs/',
					$utm_params['docs'] ?? []
				),
				'text'   => esc_html__( 'Docs', 'wpforms-lite' ),
				'target' => '_blank',
				'icon'   => '<svg xmlns="http://www.w3.org/2000/svg" width="13" height="17" viewBox="0 0 13 17" fill="none"><path d="M2.33765 14.5854H10.3376C10.5876 14.5854 10.8376 14.3667 10.8376 14.0854V5.08545H8.33765C7.77515 5.08545 7.33765 4.64795 7.33765 4.08545V1.58545H2.33765C2.0564 1.58545 1.83765 1.83545 1.83765 2.08545V14.0854C1.83765 14.3667 2.0564 14.5854 2.33765 14.5854ZM2.33765 0.0854492H7.4939C8.02515 0.0854492 8.52515 0.304199 8.90015 0.679199L11.7439 3.52295C12.1189 3.89795 12.3376 4.39795 12.3376 4.9292V14.0854C12.3376 15.2104 11.4314 16.0854 10.3376 16.0854H2.33765C1.21265 16.0854 0.337646 15.2104 0.337646 14.0854V2.08545C0.337646 0.991699 1.21265 0.0854492 2.33765 0.0854492ZM4.08765 8.08545H8.58765C8.9939 8.08545 9.33765 8.4292 9.33765 8.83545C9.33765 9.27295 8.9939 9.58545 8.58765 9.58545H4.08765C3.65015 9.58545 3.33765 9.27295 3.33765 8.83545C3.33765 8.4292 3.65015 8.08545 4.08765 8.08545ZM4.08765 11.0854H8.58765C8.9939 11.0854 9.33765 11.4292 9.33765 11.8354C9.33765 12.2729 8.9939 12.5854 8.58765 12.5854H4.08765C3.65015 12.5854 3.33765 12.2729 3.33765 11.8354C3.33765 11.4292 3.65015 11.0854 4.08765 11.0854Z" fill="#646970"/></svg>',
			],
			'videos'    => [
				'url'    => 'https://www.youtube.com/@wpforms/videos',
				'text'   => esc_html__( 'Videos', 'wpforms-lite' ),
				'target' => '_blank',
				'icon'   => '<svg xmlns="http://www.w3.org/2000/svg" width="17" height="17" viewBox="0 0 17 17" fill="none"><path d="M14.8376 8.06982C14.8376 5.75732 13.5876 3.63232 11.5876 2.44482C9.5564 1.28857 7.08765 1.28857 5.08765 2.44482C3.0564 3.63232 1.83765 5.75732 1.83765 8.06982C1.83765 10.4136 3.0564 12.5386 5.08765 13.7261C7.08765 14.8823 9.5564 14.8823 11.5876 13.7261C13.5876 12.5386 14.8376 10.4136 14.8376 8.06982ZM0.337646 8.06982C0.337646 5.22607 1.83765 2.60107 4.33765 1.16357C6.8064 -0.273926 9.83765 -0.273926 12.3376 1.16357C14.8064 2.60107 16.3376 5.22607 16.3376 8.06982C16.3376 10.9448 14.8064 13.5698 12.3376 15.0073C9.83765 16.4448 6.8064 16.4448 4.33765 15.0073C1.83765 13.5698 0.337646 10.9448 0.337646 8.06982ZM6.21265 4.69482C6.4314 4.53857 6.7439 4.53857 6.96265 4.69482L11.4626 7.44482C11.6814 7.56982 11.8376 7.81982 11.8376 8.10107C11.8376 8.35107 11.6814 8.60107 11.4626 8.72607L6.96265 11.4761C6.7439 11.6323 6.4314 11.6323 6.21265 11.5073C5.96265 11.3511 5.83765 11.1011 5.83765 10.8511V5.35107C5.83765 5.06982 5.96265 4.81982 6.21265 4.69482Z" fill="#646970"/></svg>',
			],
			'support'   => [
				'url'    => wpforms()->is_pro() ?
					self::get_utm_link(
						'https://wpforms.com/account/support/',
						$utm_params['support'] ?? []
					) : 'https://wordpress.org/support/plugin/wpforms-lite/',
				'text'   => wpforms()->is_pro() ? esc_html__( 'Support', 'wpforms-lite' ) : esc_html__( 'Support Forum', 'wpforms-lite' ),
				'target' => '_blank',
				'icon'   => wpforms()->is_pro() ?
					'<svg xmlns="http://www.w3.org/2000/svg" width="17" height="17" viewBox="0 0 17 17" fill="none"><path d="M14.8376 8.06982C14.8376 5.75732 13.5876 3.63232 11.5876 2.44482C9.5564 1.28857 7.08765 1.28857 5.08765 2.44482C3.0564 3.63232 1.83765 5.75732 1.83765 8.06982C1.83765 10.4136 3.0564 12.5386 5.08765 13.7261C7.08765 14.8823 9.5564 14.8823 11.5876 13.7261C13.5876 12.5386 14.8376 10.4136 14.8376 8.06982ZM0.337646 8.06982C0.337646 5.22607 1.83765 2.60107 4.33765 1.16357C6.8064 -0.273926 9.83765 -0.273926 12.3376 1.16357C14.8064 2.60107 16.3376 5.22607 16.3376 8.06982C16.3376 10.9448 14.8064 13.5698 12.3376 15.0073C9.83765 16.4448 6.8064 16.4448 4.33765 15.0073C1.83765 13.5698 0.337646 10.9448 0.337646 8.06982ZM5.6189 5.25732C5.8689 4.53857 6.52515 4.06982 7.27515 4.06982H9.08765C10.1814 4.06982 11.0876 4.97607 11.0876 6.06982C11.0876 6.75732 10.6814 7.41357 10.0876 7.75732L9.08765 8.35107C9.0564 8.75732 8.7439 9.06982 8.33765 9.06982C7.90015 9.06982 7.58765 8.75732 7.58765 8.31982V7.91357C7.58765 7.63232 7.71265 7.38232 7.96265 7.25732L9.33765 6.47607C9.4939 6.38232 9.58765 6.22607 9.58765 6.06982C9.58765 5.78857 9.3689 5.60107 9.08765 5.60107H7.27515C7.1814 5.60107 7.08765 5.66357 7.0564 5.75732L7.02515 5.78857C6.90015 6.19482 6.46265 6.38232 6.08765 6.25732C5.6814 6.10107 5.4939 5.66357 5.6189 5.28857V5.25732ZM7.33765 11.0698C7.33765 10.5386 7.77515 10.0698 8.33765 10.0698C8.8689 10.0698 9.33765 10.5386 9.33765 11.0698C9.33765 11.6323 8.8689 12.0698 8.33765 12.0698C7.77515 12.0698 7.33765 11.6323 7.33765 11.0698Z" fill="#646970"/></svg>' :
					'<svg xmlns="http://www.w3.org/2000/svg" width="20" height="16" viewBox="0 0 20 16" fill="none"><path d="M3.09554 9.83838C3.06625 9.89697 3.03695 9.95557 2.97836 9.98486C2.91976 10.1313 2.83187 10.2485 2.77328 10.395C3.27133 10.2778 3.74008 10.0728 4.20883 9.86768C4.35531 9.80908 4.4725 9.75049 4.58968 9.69189C4.88265 9.54541 5.17562 9.51611 5.49789 9.57471C5.84945 9.6333 6.20101 9.6626 6.61117 9.6626C9.42367 9.6626 11.2987 7.7876 11.2987 5.9126C11.2987 4.06689 9.42367 2.1626 6.61117 2.1626C3.76937 2.1626 1.92367 4.06689 1.92367 5.9126C1.92367 6.73291 2.24593 7.52393 2.86117 8.19775C3.30062 8.63721 3.38851 9.28174 3.09554 9.83838ZM6.61117 11.0688C6.11312 11.0688 5.67367 11.0396 5.20492 10.9517C5.08773 11.0103 4.94125 11.0981 4.79476 11.1567C3.74008 11.6255 2.48031 12.0063 1.22054 12.0063C0.927575 12.0063 0.663904 11.8599 0.546716 11.5962C0.458825 11.3325 0.488122 11.0396 0.6932 10.8345C1.10336 10.3657 1.45492 9.83838 1.77718 9.31104C1.80648 9.25244 1.83578 9.19385 1.83578 9.16455C1.01547 8.28564 0.517419 7.14307 0.517419 5.9126C0.517419 3.0708 3.24203 0.756348 6.61117 0.756348C9.95101 0.756348 12.7049 3.0708 12.7049 5.9126C12.7049 8.78369 9.95101 11.0688 6.61117 11.0688ZM13.1737 14.8188C10.7713 14.8188 8.69125 13.647 7.69515 11.9478C8.1932 11.8599 8.69125 11.7427 9.16 11.5962C9.92172 12.6509 11.3573 13.4126 13.1444 13.4126C13.5252 13.4126 13.9061 13.3833 14.2577 13.3247C14.5799 13.2661 14.8729 13.2954 15.1659 13.4419C15.283 13.5005 15.4002 13.5591 15.5467 13.6177C16.0155 13.8228 16.4842 13.9985 16.9823 14.145C16.9237 13.9985 16.8358 13.8813 16.7772 13.7349C16.7479 13.6763 16.6893 13.6177 16.66 13.5591C16.3963 13.0317 16.4842 12.3872 16.8944 11.9478C17.5096 11.2739 17.8612 10.4829 17.8612 9.6626C17.8612 7.93408 16.1912 6.14697 13.6424 5.94189V5.9126C13.6424 5.44385 13.5545 4.9751 13.4373 4.53564C16.6893 4.65283 19.2674 6.90869 19.2674 9.6626C19.2674 10.8931 18.7401 12.0356 17.9198 12.9146C17.9198 12.9731 17.9491 13.0024 17.9784 13.061C18.3006 13.5884 18.6522 14.1157 19.0623 14.5845C19.2674 14.7896 19.2967 15.0825 19.2088 15.3462C19.0916 15.6099 18.828 15.7563 18.5643 15.7563C17.3045 15.7563 16.0155 15.3755 14.9608 14.9067C14.8143 14.8481 14.6678 14.7603 14.5506 14.7017C14.0819 14.7896 13.6424 14.8188 13.1737 14.8188Z" fill="#50575E"/></svg>',
			],
			'whats-new' => [
				'text'  => esc_html__( 'What’s New', 'wpforms-lite' ),
				'class' => 'wpforms-splash-modal-open',
				'icon'  => '<svg xmlns="http://www.w3.org/2000/svg" width="17" height="20" viewBox="0 0 17 20" fill="none"><path d="M13.6014 1.63137L14.7985 6.09878C15.4146 6.22486 16.0232 6.80589 16.2497 7.65107C16.4762 8.49626 16.2477 9.33393 15.7771 9.75119L16.9661 14.1884C17.0712 14.5808 16.9268 15.0077 16.605 15.2557C16.2833 15.5037 15.8364 15.5264 15.4919 15.3275L13.8079 14.3552C11.9708 13.2946 9.79038 13.0053 7.73779 13.5553L7.4963 13.62L8.53158 17.4837C8.67717 18.027 8.33762 18.571 7.82447 18.7085L5.89262 19.2261C5.34929 19.3717 4.81346 19.0623 4.66788 18.519L3.6326 14.6553C2.54594 14.9465 1.47428 14.3277 1.18311 13.2411L0.406655 10.3433C0.123572 9.28681 0.734202 8.18497 1.82087 7.8938L5.92605 6.79382C7.97865 6.24383 9.7223 4.9031 10.783 3.06598L11.7633 1.41214C11.9622 1.06769 12.3605 0.863896 12.7632 0.917765C13.1659 0.971634 13.5044 1.26915 13.6014 1.63137ZM12.2924 4.47327C10.9482 6.58047 8.85851 8.07862 6.44369 8.72567L6.20221 8.79037L6.97867 11.6882L7.22015 11.6234C9.63496 10.9764 12.2019 11.2592 14.4195 12.412L12.2924 4.47327Z" fill="#646970"/></svg>',
			],
		];
	}

	/**
	 * Get UTM link.
	 *
	 * @since 1.9.7
	 *
	 * @param string $url        The URL to which UTM parameters will be added.
	 * @param array  $utm_params UTM parameters to append to the URL.
	 *
	 * @return string
	 */
	private static function get_utm_link( string $url, array $utm_params ): string {

		return wpforms_utm_link(
			$url,
			$utm_params['medium'] ?? '',
			$utm_params['content'] ?? '',
			$utm_params['term'] ?? ''
		);
	}
}
